<?php
declare(strict_types=1);

namespace ZWEI14\T3template\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Core\Database\ConnectionPool;

use TYPO3\CMS\Core\Database\Query\Restriction\StartTimeRestriction;
use TYPO3\CMS\Core\Database\Query\Restriction\EndTimeRestriction;

/**
 * This task checks if the user has registered 10 Years (3650 days) ago
 */
class DeleteUsersNeverLoggedInForEightMonthsOrNotLoggedInForTwelveMonthsCommand extends Command
{

    private $stderr = '';

    protected $neverLoggedInSinceDays = (8*31);

    protected $notLoggedInSinceDays = (365);

    /**
     * Configures the current command.
     */
    protected function configure()
    {
        parent::configure();
        $this->setDescription('Delete users that never logged in for eight months or not logged in for twelve months.')
        ->addOption('dry-run', null, InputOption::VALUE_NONE, 'Don`t actually delete users.');
    }

    /**
     * Executes the current command.
     *
     * This method is not abstract because you can use this class
     * as a concrete class. In this case, instead of defining the
     * execute() method, you set the code to execute by passing
     * a Closure to the setCode() method.
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return int|null null or 0 if everything went fine, or an error code
     *
     * @throws \TYPO3\CMS\Core\Cache\Exception\NoSuchCacheException
     *
     * @see setCode()
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $dryRun = ($input->getOption('dry-run') == 1 ? true : false);

        $io = new SymfonyStyle($input, $output);

        $queryBuilder = GeneralUtility::makeInstance(ConnectionPool::class)->getQueryBuilderForTable('fe_users');
        $queryBuilder
            ->getRestrictions()
            ->removeAll();
        //     // ->add(GeneralUtility::makeInstance(StartTimeRestriction::class))
        //     // ->add(GeneralUtility::makeInstance(EndTimeRestriction::class));
        //     // ->removeByType(DeletedRestriction::class)
        //     // ->removeByType(HiddenRestriction::class);
        $result = $queryBuilder
            ->select('fe_users.uid','fe_users.deleted','fe_users.disable','fe_users.username','fe_users.first_name','fe_users.last_name','fe_users.email','fe_users.mail_after_approval','fe_users.mail_after_manual_approval')
            ->from('fe_users')
            ->where(
                $queryBuilder->expr()->lte('fe_users.crdate', $queryBuilder->createNamedParameter((time() - $this->neverLoggedInSinceDays*24*60*60), \PDO::PARAM_INT)),
                $queryBuilder->expr()->eq('fe_users.lastlogin', $queryBuilder->createNamedParameter(0, \PDO::PARAM_INT)),
                $queryBuilder->expr()->notLike(
                    'fe_users.email',
                    $queryBuilder->createNamedParameter('%' . $queryBuilder->escapeLikeWildcards('zwei14.de'))
                ),
                $queryBuilder->expr()->eq('fe_users.disable', $queryBuilder->createNamedParameter(0, \PDO::PARAM_INT)),
                $queryBuilder->expr()->eq('fe_users.deleted', $queryBuilder->createNamedParameter(0, \PDO::PARAM_INT))
            )
            ->orWhere(
                $queryBuilder->expr()->andX(
                    $queryBuilder->expr()->lte('fe_users.lastlogin',$queryBuilder->createNamedParameter((time() - $this->notLoggedInSinceDays*24*60*60), \PDO::PARAM_INT)),
                    $queryBuilder->expr()->gt('fe_users.lastlogin', $queryBuilder->createNamedParameter(0, \PDO::PARAM_INT)),
                    $queryBuilder->expr()->notLike(
                        'fe_users.email',
                        $queryBuilder->createNamedParameter('%' . $queryBuilder->escapeLikeWildcards('zwei14.de'))
                    ),
                    $queryBuilder->expr()->eq('fe_users.disable', $queryBuilder->createNamedParameter(0, \PDO::PARAM_INT)),
                    $queryBuilder->expr()->eq('fe_users.deleted', $queryBuilder->createNamedParameter(0, \PDO::PARAM_INT))
                )
            )
            ->execute()
            ->fetchAllAssociative();

        // debug($queryBuilder->getSql());
        // debug($result);
        // die;

        if(empty($result)) {

            $io->success('Nothing to do');
            $this->appendToStderr('[SUCCESS] Nothing to do.');
            $this->writeToPhpStderr($this->getStderr());
            return 0;

        }

        $table = new Table($output);
        $tableRows = [];
            foreach ($result as $key => $values) {

                $printValues = [
                    'uid' => $values['uid'],
                    'email' => mb_substr($values['email'], 0, 3, 'UTF-8') . '...' . mb_substr($values['email'], -6, 6, 'UTF-8'),
                    'deleted' => $values['deleted'],
                    'success' => 'no (error)',
                ];

                if ($key === 0) {
                    $table
                        ->setHeaders(array_keys($printValues));
                }

                try {

                    $printValues['success'] = 'no (dry-run)';
                    if( ! $dryRun) {
                        // And finally delete user
                        $queryBuilder
                        ->update('fe_users')
                        ->where(
                            $queryBuilder->expr()->eq('fe_users.uid', $queryBuilder->createNamedParameter($values['uid'], \PDO::PARAM_INT))
                        )
                        ->set('deleted', 1)
                        ->execute();
                        $printValues['deleted'] = 1;
                        $printValues['success'] = 'yes';
                    }
                    
                    $tableRows[] = $printValues;

                } catch (\Exception $e) {
                    $tableRows[] = $printValues;
                }

            }

        $table->setRows($tableRows);
        $table->render();

        $io->success('Done');
        foreach($tableRows as $row) {
            $this->appendToStderr(implode(" | ", $row));
        }
        
        $this->appendToStderr('[SUCCESS] Done.');
        $this->writeToPhpStderr($this->getStderr());
        return 0;
    }

    /**
     * @param string $value
     */
    public function appendToStderr (string $value): void {

        if ($this->stderr === '') {
            $this->stderr .= $value;
            return;
        }

        $this->stderr .= PHP_EOL . $value;

    }

    /**
     * @return string
     */
    public function getStderr(): string
    {
        return $this->stderr;
    }

    /**
     * @param string $value
     */
    public function setStderr(string $value): void
    {
        $this->stderr = $value;
    }

    /**
     * @param string $value
     */
    function writeToPhpStderr(string $value): void
    {
        $stderr = fopen('php://stderr', 'w');
        fwrite($stderr, $value);
        fclose($stderr);
    }
}
